/***************************************************************************//**
* \file SPIM_UART.c
* \version 4.0
*
* \brief
*  This file provides the source code to the API for the SCB Component in
*  UART mode.
*
* Note:
*
*******************************************************************************
* \copyright
* Copyright 2013-2017, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions,
* disclaimers, and limitations in the end user license agreement accompanying
* the software package with which this file was provided.
*******************************************************************************/

#include "SPIM_PVT.h"
#include "SPIM_SPI_UART_PVT.h"
#include "cyapicallbacks.h"

#if (SPIM_UART_WAKE_ENABLE_CONST && SPIM_UART_RX_WAKEUP_IRQ)
    /**
    * \addtogroup group_globals
    * \{
    */
    /** This global variable determines whether to enable Skip Start
    * functionality when SPIM_Sleep() function is called:
    * 0 – disable, other values – enable. Default value is 1.
    * It is only available when Enable wakeup from Deep Sleep Mode is enabled.
    */
    uint8 SPIM_skipStart = 1u;
    /** \} globals */
#endif /* (SPIM_UART_WAKE_ENABLE_CONST && SPIM_UART_RX_WAKEUP_IRQ) */

#if(SPIM_SCB_MODE_UNCONFIG_CONST_CFG)

    /***************************************
    *  Configuration Structure Initialization
    ***************************************/

    const SPIM_UART_INIT_STRUCT SPIM_configUart =
    {
        SPIM_UART_SUB_MODE,
        SPIM_UART_DIRECTION,
        SPIM_UART_DATA_BITS_NUM,
        SPIM_UART_PARITY_TYPE,
        SPIM_UART_STOP_BITS_NUM,
        SPIM_UART_OVS_FACTOR,
        SPIM_UART_IRDA_LOW_POWER,
        SPIM_UART_MEDIAN_FILTER_ENABLE,
        SPIM_UART_RETRY_ON_NACK,
        SPIM_UART_IRDA_POLARITY,
        SPIM_UART_DROP_ON_PARITY_ERR,
        SPIM_UART_DROP_ON_FRAME_ERR,
        SPIM_UART_WAKE_ENABLE,
        0u,
        NULL,
        0u,
        NULL,
        SPIM_UART_MP_MODE_ENABLE,
        SPIM_UART_MP_ACCEPT_ADDRESS,
        SPIM_UART_MP_RX_ADDRESS,
        SPIM_UART_MP_RX_ADDRESS_MASK,
        (uint32) SPIM_SCB_IRQ_INTERNAL,
        SPIM_UART_INTR_RX_MASK,
        SPIM_UART_RX_TRIGGER_LEVEL,
        SPIM_UART_INTR_TX_MASK,
        SPIM_UART_TX_TRIGGER_LEVEL,
        (uint8) SPIM_UART_BYTE_MODE_ENABLE,
        (uint8) SPIM_UART_CTS_ENABLE,
        (uint8) SPIM_UART_CTS_POLARITY,
        (uint8) SPIM_UART_RTS_POLARITY,
        (uint8) SPIM_UART_RTS_FIFO_LEVEL,
        (uint8) SPIM_UART_RX_BREAK_WIDTH
    };


    /*******************************************************************************
    * Function Name: SPIM_UartInit
    ****************************************************************************//**
    *
    *  Configures the SPIM for UART operation.
    *
    *  This function is intended specifically to be used when the SPIM
    *  configuration is set to “Unconfigured SPIM” in the customizer.
    *  After initializing the SPIM in UART mode using this function,
    *  the component can be enabled using the SPIM_Start() or
    * SPIM_Enable() function.
    *  This function uses a pointer to a structure that provides the configuration
    *  settings. This structure contains the same information that would otherwise
    *  be provided by the customizer settings.
    *
    *  \param config: pointer to a structure that contains the following list of
    *   fields. These fields match the selections available in the customizer.
    *   Refer to the customizer for further description of the settings.
    *
    *******************************************************************************/
    void SPIM_UartInit(const SPIM_UART_INIT_STRUCT *config)
    {
        uint32 pinsConfig;

        if (NULL == config)
        {
            CYASSERT(0u != 0u); /* Halt execution due to bad function parameter */
        }
        else
        {
            /* Get direction to configure UART pins: TX, RX or TX+RX */
            pinsConfig  = config->direction;

        #if !(SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1)
            /* Add RTS and CTS pins to configure */
            pinsConfig |= (0u != config->rtsRxFifoLevel) ? (SPIM_UART_RTS_PIN_ENABLE) : (0u);
            pinsConfig |= (0u != config->enableCts)      ? (SPIM_UART_CTS_PIN_ENABLE) : (0u);
        #endif /* !(SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1) */

            /* Configure pins */
            SPIM_SetPins(SPIM_SCB_MODE_UART, config->mode, pinsConfig);

            /* Store internal configuration */
            SPIM_scbMode       = (uint8) SPIM_SCB_MODE_UART;
            SPIM_scbEnableWake = (uint8) config->enableWake;
            SPIM_scbEnableIntr = (uint8) config->enableInterrupt;

            /* Set RX direction internal variables */
            SPIM_rxBuffer      =         config->rxBuffer;
            SPIM_rxDataBits    = (uint8) config->dataBits;
            SPIM_rxBufferSize  =         config->rxBufferSize;

            /* Set TX direction internal variables */
            SPIM_txBuffer      =         config->txBuffer;
            SPIM_txDataBits    = (uint8) config->dataBits;
            SPIM_txBufferSize  =         config->txBufferSize;

            /* Configure UART interface */
            if(SPIM_UART_MODE_IRDA == config->mode)
            {
                /* OVS settings: IrDA */
                SPIM_CTRL_REG  = ((0u != config->enableIrdaLowPower) ?
                                                (SPIM_UART_GET_CTRL_OVS_IRDA_LP(config->oversample)) :
                                                (SPIM_CTRL_OVS_IRDA_OVS16));
            }
            else
            {
                /* OVS settings: UART and SmartCard */
                SPIM_CTRL_REG  = SPIM_GET_CTRL_OVS(config->oversample);
            }

            SPIM_CTRL_REG     |= SPIM_GET_CTRL_BYTE_MODE  (config->enableByteMode)      |
                                             SPIM_GET_CTRL_ADDR_ACCEPT(config->multiprocAcceptAddr) |
                                             SPIM_CTRL_UART;

            /* Configure sub-mode: UART, SmartCard or IrDA */
            SPIM_UART_CTRL_REG = SPIM_GET_UART_CTRL_MODE(config->mode);

            /* Configure RX direction */
            SPIM_UART_RX_CTRL_REG = SPIM_GET_UART_RX_CTRL_MODE(config->stopBits)              |
                                        SPIM_GET_UART_RX_CTRL_POLARITY(config->enableInvertedRx)          |
                                        SPIM_GET_UART_RX_CTRL_MP_MODE(config->enableMultiproc)            |
                                        SPIM_GET_UART_RX_CTRL_DROP_ON_PARITY_ERR(config->dropOnParityErr) |
                                        SPIM_GET_UART_RX_CTRL_DROP_ON_FRAME_ERR(config->dropOnFrameErr)   |
                                        SPIM_GET_UART_RX_CTRL_BREAK_WIDTH(config->breakWidth);

            if(SPIM_UART_PARITY_NONE != config->parity)
            {
               SPIM_UART_RX_CTRL_REG |= SPIM_GET_UART_RX_CTRL_PARITY(config->parity) |
                                                    SPIM_UART_RX_CTRL_PARITY_ENABLED;
            }

            SPIM_RX_CTRL_REG      = SPIM_GET_RX_CTRL_DATA_WIDTH(config->dataBits)       |
                                                SPIM_GET_RX_CTRL_MEDIAN(config->enableMedianFilter) |
                                                SPIM_GET_UART_RX_CTRL_ENABLED(config->direction);

            SPIM_RX_FIFO_CTRL_REG = SPIM_GET_RX_FIFO_CTRL_TRIGGER_LEVEL(config->rxTriggerLevel);

            /* Configure MP address */
            SPIM_RX_MATCH_REG     = SPIM_GET_RX_MATCH_ADDR(config->multiprocAddr) |
                                                SPIM_GET_RX_MATCH_MASK(config->multiprocAddrMask);

            /* Configure RX direction */
            SPIM_UART_TX_CTRL_REG = SPIM_GET_UART_TX_CTRL_MODE(config->stopBits) |
                                                SPIM_GET_UART_TX_CTRL_RETRY_NACK(config->enableRetryNack);

            if(SPIM_UART_PARITY_NONE != config->parity)
            {
               SPIM_UART_TX_CTRL_REG |= SPIM_GET_UART_TX_CTRL_PARITY(config->parity) |
                                                    SPIM_UART_TX_CTRL_PARITY_ENABLED;
            }

            SPIM_TX_CTRL_REG      = SPIM_GET_TX_CTRL_DATA_WIDTH(config->dataBits)    |
                                                SPIM_GET_UART_TX_CTRL_ENABLED(config->direction);

            SPIM_TX_FIFO_CTRL_REG = SPIM_GET_TX_FIFO_CTRL_TRIGGER_LEVEL(config->txTriggerLevel);

        #if !(SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1)
            SPIM_UART_FLOW_CTRL_REG = SPIM_GET_UART_FLOW_CTRL_CTS_ENABLE(config->enableCts) | \
                                            SPIM_GET_UART_FLOW_CTRL_CTS_POLARITY (config->ctsPolarity)  | \
                                            SPIM_GET_UART_FLOW_CTRL_RTS_POLARITY (config->rtsPolarity)  | \
                                            SPIM_GET_UART_FLOW_CTRL_TRIGGER_LEVEL(config->rtsRxFifoLevel);
        #endif /* !(SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1) */

            /* Configure interrupt with UART handler but do not enable it */
            CyIntDisable    (SPIM_ISR_NUMBER);
            CyIntSetPriority(SPIM_ISR_NUMBER, SPIM_ISR_PRIORITY);
            (void) CyIntSetVector(SPIM_ISR_NUMBER, &SPIM_SPI_UART_ISR);

            /* Configure WAKE interrupt */
        #if(SPIM_UART_RX_WAKEUP_IRQ)
            CyIntDisable    (SPIM_RX_WAKE_ISR_NUMBER);
            CyIntSetPriority(SPIM_RX_WAKE_ISR_NUMBER, SPIM_RX_WAKE_ISR_PRIORITY);
            (void) CyIntSetVector(SPIM_RX_WAKE_ISR_NUMBER, &SPIM_UART_WAKEUP_ISR);
        #endif /* (SPIM_UART_RX_WAKEUP_IRQ) */

            /* Configure interrupt sources */
            SPIM_INTR_I2C_EC_MASK_REG = SPIM_NO_INTR_SOURCES;
            SPIM_INTR_SPI_EC_MASK_REG = SPIM_NO_INTR_SOURCES;
            SPIM_INTR_SLAVE_MASK_REG  = SPIM_NO_INTR_SOURCES;
            SPIM_INTR_MASTER_MASK_REG = SPIM_NO_INTR_SOURCES;
            SPIM_INTR_RX_MASK_REG     = config->rxInterruptMask;
            SPIM_INTR_TX_MASK_REG     = config->txInterruptMask;

            /* Configure TX interrupt sources to restore. */
            SPIM_IntrTxMask = LO16(SPIM_INTR_TX_MASK_REG);

            /* Clear RX buffer indexes */
            SPIM_rxBufferHead     = 0u;
            SPIM_rxBufferTail     = 0u;
            SPIM_rxBufferOverflow = 0u;

            /* Clear TX buffer indexes */
            SPIM_txBufferHead = 0u;
            SPIM_txBufferTail = 0u;
        }
    }

#else

    /*******************************************************************************
    * Function Name: SPIM_UartInit
    ****************************************************************************//**
    *
    *  Configures the SCB for the UART operation.
    *
    *******************************************************************************/
    void SPIM_UartInit(void)
    {
        /* Configure UART interface */
        SPIM_CTRL_REG = SPIM_UART_DEFAULT_CTRL;

        /* Configure sub-mode: UART, SmartCard or IrDA */
        SPIM_UART_CTRL_REG = SPIM_UART_DEFAULT_UART_CTRL;

        /* Configure RX direction */
        SPIM_UART_RX_CTRL_REG = SPIM_UART_DEFAULT_UART_RX_CTRL;
        SPIM_RX_CTRL_REG      = SPIM_UART_DEFAULT_RX_CTRL;
        SPIM_RX_FIFO_CTRL_REG = SPIM_UART_DEFAULT_RX_FIFO_CTRL;
        SPIM_RX_MATCH_REG     = SPIM_UART_DEFAULT_RX_MATCH_REG;

        /* Configure TX direction */
        SPIM_UART_TX_CTRL_REG = SPIM_UART_DEFAULT_UART_TX_CTRL;
        SPIM_TX_CTRL_REG      = SPIM_UART_DEFAULT_TX_CTRL;
        SPIM_TX_FIFO_CTRL_REG = SPIM_UART_DEFAULT_TX_FIFO_CTRL;

    #if !(SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1)
        SPIM_UART_FLOW_CTRL_REG = SPIM_UART_DEFAULT_FLOW_CTRL;
    #endif /* !(SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1) */

        /* Configure interrupt with UART handler but do not enable it */
    #if(SPIM_SCB_IRQ_INTERNAL)
        CyIntDisable    (SPIM_ISR_NUMBER);
        CyIntSetPriority(SPIM_ISR_NUMBER, SPIM_ISR_PRIORITY);
        (void) CyIntSetVector(SPIM_ISR_NUMBER, &SPIM_SPI_UART_ISR);
    #endif /* (SPIM_SCB_IRQ_INTERNAL) */

        /* Configure WAKE interrupt */
    #if(SPIM_UART_RX_WAKEUP_IRQ)
        CyIntDisable    (SPIM_RX_WAKE_ISR_NUMBER);
        CyIntSetPriority(SPIM_RX_WAKE_ISR_NUMBER, SPIM_RX_WAKE_ISR_PRIORITY);
        (void) CyIntSetVector(SPIM_RX_WAKE_ISR_NUMBER, &SPIM_UART_WAKEUP_ISR);
    #endif /* (SPIM_UART_RX_WAKEUP_IRQ) */

        /* Configure interrupt sources */
        SPIM_INTR_I2C_EC_MASK_REG = SPIM_UART_DEFAULT_INTR_I2C_EC_MASK;
        SPIM_INTR_SPI_EC_MASK_REG = SPIM_UART_DEFAULT_INTR_SPI_EC_MASK;
        SPIM_INTR_SLAVE_MASK_REG  = SPIM_UART_DEFAULT_INTR_SLAVE_MASK;
        SPIM_INTR_MASTER_MASK_REG = SPIM_UART_DEFAULT_INTR_MASTER_MASK;
        SPIM_INTR_RX_MASK_REG     = SPIM_UART_DEFAULT_INTR_RX_MASK;
        SPIM_INTR_TX_MASK_REG     = SPIM_UART_DEFAULT_INTR_TX_MASK;

        /* Configure TX interrupt sources to restore. */
        SPIM_IntrTxMask = LO16(SPIM_INTR_TX_MASK_REG);

    #if(SPIM_INTERNAL_RX_SW_BUFFER_CONST)
        SPIM_rxBufferHead     = 0u;
        SPIM_rxBufferTail     = 0u;
        SPIM_rxBufferOverflow = 0u;
    #endif /* (SPIM_INTERNAL_RX_SW_BUFFER_CONST) */

    #if(SPIM_INTERNAL_TX_SW_BUFFER_CONST)
        SPIM_txBufferHead = 0u;
        SPIM_txBufferTail = 0u;
    #endif /* (SPIM_INTERNAL_TX_SW_BUFFER_CONST) */
    }
#endif /* (SPIM_SCB_MODE_UNCONFIG_CONST_CFG) */


/*******************************************************************************
* Function Name: SPIM_UartPostEnable
****************************************************************************//**
*
*  Restores HSIOM settings for the UART output pins (TX and/or RTS) to be
*  controlled by the SCB UART.
*
*******************************************************************************/
void SPIM_UartPostEnable(void)
{
#if (SPIM_SCB_MODE_UNCONFIG_CONST_CFG)
    #if (SPIM_TX_SDA_MISO_PIN)
        if (SPIM_CHECK_TX_SDA_MISO_PIN_USED)
        {
            /* Set SCB UART to drive the output pin */
            SPIM_SET_HSIOM_SEL(SPIM_TX_SDA_MISO_HSIOM_REG, SPIM_TX_SDA_MISO_HSIOM_MASK,
                                           SPIM_TX_SDA_MISO_HSIOM_POS, SPIM_TX_SDA_MISO_HSIOM_SEL_UART);
        }
    #endif /* (SPIM_TX_SDA_MISO_PIN_PIN) */

    #if !(SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1)
        #if (SPIM_SS0_PIN)
            if (SPIM_CHECK_SS0_PIN_USED)
            {
                /* Set SCB UART to drive the output pin */
                SPIM_SET_HSIOM_SEL(SPIM_SS0_HSIOM_REG, SPIM_SS0_HSIOM_MASK,
                                               SPIM_SS0_HSIOM_POS, SPIM_SS0_HSIOM_SEL_UART);
            }
        #endif /* (SPIM_SS0_PIN) */
    #endif /* !(SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1) */

#else
    #if (SPIM_UART_TX_PIN)
         /* Set SCB UART to drive the output pin */
        SPIM_SET_HSIOM_SEL(SPIM_TX_HSIOM_REG, SPIM_TX_HSIOM_MASK,
                                       SPIM_TX_HSIOM_POS, SPIM_TX_HSIOM_SEL_UART);
    #endif /* (SPIM_UART_TX_PIN) */

    #if (SPIM_UART_RTS_PIN)
        /* Set SCB UART to drive the output pin */
        SPIM_SET_HSIOM_SEL(SPIM_RTS_HSIOM_REG, SPIM_RTS_HSIOM_MASK,
                                       SPIM_RTS_HSIOM_POS, SPIM_RTS_HSIOM_SEL_UART);
    #endif /* (SPIM_UART_RTS_PIN) */
#endif /* (SPIM_SCB_MODE_UNCONFIG_CONST_CFG) */

    /* Restore TX interrupt sources. */
    SPIM_SetTxInterruptMode(SPIM_IntrTxMask);
}


/*******************************************************************************
* Function Name: SPIM_UartStop
****************************************************************************//**
*
*  Changes the HSIOM settings for the UART output pins (TX and/or RTS) to keep
*  them inactive after the block is disabled. The output pins are controlled by
*  the GPIO data register. Also, the function disables the skip start feature
*  to not cause it to trigger after the component is enabled.
*
*******************************************************************************/
void SPIM_UartStop(void)
{
#if(SPIM_SCB_MODE_UNCONFIG_CONST_CFG)
    #if (SPIM_TX_SDA_MISO_PIN)
        if (SPIM_CHECK_TX_SDA_MISO_PIN_USED)
        {
            /* Set GPIO to drive output pin */
            SPIM_SET_HSIOM_SEL(SPIM_TX_SDA_MISO_HSIOM_REG, SPIM_TX_SDA_MISO_HSIOM_MASK,
                                           SPIM_TX_SDA_MISO_HSIOM_POS, SPIM_TX_SDA_MISO_HSIOM_SEL_GPIO);
        }
    #endif /* (SPIM_TX_SDA_MISO_PIN_PIN) */

    #if !(SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1)
        #if (SPIM_SS0_PIN)
            if (SPIM_CHECK_SS0_PIN_USED)
            {
                /* Set output pin state after block is disabled */
                SPIM_spi_ss0_Write(SPIM_GET_UART_RTS_INACTIVE);

                /* Set GPIO to drive output pin */
                SPIM_SET_HSIOM_SEL(SPIM_SS0_HSIOM_REG, SPIM_SS0_HSIOM_MASK,
                                               SPIM_SS0_HSIOM_POS, SPIM_SS0_HSIOM_SEL_GPIO);
            }
        #endif /* (SPIM_SS0_PIN) */
    #endif /* !(SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1) */

#else
    #if (SPIM_UART_TX_PIN)
        /* Set GPIO to drive output pin */
        SPIM_SET_HSIOM_SEL(SPIM_TX_HSIOM_REG, SPIM_TX_HSIOM_MASK,
                                       SPIM_TX_HSIOM_POS, SPIM_TX_HSIOM_SEL_GPIO);
    #endif /* (SPIM_UART_TX_PIN) */

    #if (SPIM_UART_RTS_PIN)
        /* Set output pin state after block is disabled */
        SPIM_rts_Write(SPIM_GET_UART_RTS_INACTIVE);

        /* Set GPIO to drive output pin */
        SPIM_SET_HSIOM_SEL(SPIM_RTS_HSIOM_REG, SPIM_RTS_HSIOM_MASK,
                                       SPIM_RTS_HSIOM_POS, SPIM_RTS_HSIOM_SEL_GPIO);
    #endif /* (SPIM_UART_RTS_PIN) */

#endif /* (SPIM_SCB_MODE_UNCONFIG_CONST_CFG) */

#if (SPIM_UART_WAKE_ENABLE_CONST)
    /* Disable skip start feature used for wakeup */
    SPIM_UART_RX_CTRL_REG &= (uint32) ~SPIM_UART_RX_CTRL_SKIP_START;
#endif /* (SPIM_UART_WAKE_ENABLE_CONST) */

    /* Store TX interrupt sources (exclude level triggered). */
    SPIM_IntrTxMask = LO16(SPIM_GetTxInterruptMode() & SPIM_INTR_UART_TX_RESTORE);
}


/*******************************************************************************
* Function Name: SPIM_UartSetRxAddress
****************************************************************************//**
*
*  Sets the hardware detectable receiver address for the UART in the
*  Multiprocessor mode.
*
*  \param address: Address for hardware address detection.
*
*******************************************************************************/
void SPIM_UartSetRxAddress(uint32 address)
{
     uint32 matchReg;

    matchReg = SPIM_RX_MATCH_REG;

    matchReg &= ((uint32) ~SPIM_RX_MATCH_ADDR_MASK); /* Clear address bits */
    matchReg |= ((uint32)  (address & SPIM_RX_MATCH_ADDR_MASK)); /* Set address  */

    SPIM_RX_MATCH_REG = matchReg;
}


/*******************************************************************************
* Function Name: SPIM_UartSetRxAddressMask
****************************************************************************//**
*
*  Sets the hardware address mask for the UART in the Multiprocessor mode.
*
*  \param addressMask: Address mask.
*   - Bit value 0 – excludes bit from address comparison.
*   - Bit value 1 – the bit needs to match with the corresponding bit
*     of the address.
*
*******************************************************************************/
void SPIM_UartSetRxAddressMask(uint32 addressMask)
{
    uint32 matchReg;

    matchReg = SPIM_RX_MATCH_REG;

    matchReg &= ((uint32) ~SPIM_RX_MATCH_MASK_MASK); /* Clear address mask bits */
    matchReg |= ((uint32) (addressMask << SPIM_RX_MATCH_MASK_POS));

    SPIM_RX_MATCH_REG = matchReg;
}


#if(SPIM_UART_RX_DIRECTION)
    /*******************************************************************************
    * Function Name: SPIM_UartGetChar
    ****************************************************************************//**
    *
    *  Retrieves next data element from receive buffer.
    *  This function is designed for ASCII characters and returns a char where
    *  1 to 255 are valid characters and 0 indicates an error occurred or no data
    *  is present.
    *  - RX software buffer is disabled: Returns data element retrieved from RX
    *    FIFO.
    *  - RX software buffer is enabled: Returns data element from the software
    *    receive buffer.
    *
    *  \return
    *   Next data element from the receive buffer. ASCII character values from
    *   1 to 255 are valid. A returned zero signifies an error condition or no
    *   data available.
    *
    *  \sideeffect
    *   The errors bits may not correspond with reading characters due to
    *   RX FIFO and software buffer usage.
    *   RX software buffer is enabled: The internal software buffer overflow
    *   is not treated as an error condition.
    *   Check SPIM_rxBufferOverflow to capture that error condition.
    *
    *******************************************************************************/
    uint32 SPIM_UartGetChar(void)
    {
        uint32 rxData = 0u;

        /* Reads data only if there is data to read */
        if (0u != SPIM_SpiUartGetRxBufferSize())
        {
            rxData = SPIM_SpiUartReadRxData();
        }

        if (SPIM_CHECK_INTR_RX(SPIM_INTR_RX_ERR))
        {
            rxData = 0u; /* Error occurred: returns zero */
            SPIM_ClearRxInterruptSource(SPIM_INTR_RX_ERR);
        }

        return (rxData);
    }


    /*******************************************************************************
    * Function Name: SPIM_UartGetByte
    ****************************************************************************//**
    *
    *  Retrieves the next data element from the receive buffer, returns the
    *  received byte and error condition.
    *   - The RX software buffer is disabled: returns the data element retrieved
    *     from the RX FIFO. Undefined data will be returned if the RX FIFO is
    *     empty.
    *   - The RX software buffer is enabled: returns data element from the
    *     software receive buffer.
    *
    *  \return
    *   Bits 7-0 contain the next data element from the receive buffer and
    *   other bits contain the error condition.
    *   - SPIM_UART_RX_OVERFLOW - Attempt to write to a full
    *     receiver FIFO.
    *   - SPIM_UART_RX_UNDERFLOW    Attempt to read from an empty
    *     receiver FIFO.
    *   - SPIM_UART_RX_FRAME_ERROR - UART framing error detected.
    *   - SPIM_UART_RX_PARITY_ERROR - UART parity error detected.
    *
    *  \sideeffect
    *   The errors bits may not correspond with reading characters due to
    *   RX FIFO and software buffer usage.
    *   RX software buffer is enabled: The internal software buffer overflow
    *   is not treated as an error condition.
    *   Check SPIM_rxBufferOverflow to capture that error condition.
    *
    *******************************************************************************/
    uint32 SPIM_UartGetByte(void)
    {
        uint32 rxData;
        uint32 tmpStatus;

        if (SPIM_CHECK_RX_SW_BUFFER)
        {
            SPIM_DisableInt();
        }
        

        if (0u != SPIM_SpiUartGetRxBufferSize())
        {
            /* Enables interrupt to receive more bytes: at least one byte is in
            * buffer.
            */
            if (SPIM_CHECK_RX_SW_BUFFER)
            {
                SPIM_EnableInt();
            }
            

            /* Get received byte */
            rxData = SPIM_SpiUartReadRxData();
        }
        else
        {
            /* Reads a byte directly from RX FIFO: underflow is raised in the
            * case of empty. Otherwise the first received byte will be read.
            */
            rxData = SPIM_RX_FIFO_RD_REG;


            /* Enables interrupt to receive more bytes. */
            if (SPIM_CHECK_RX_SW_BUFFER)
            {

                /* The byte has been read from RX FIFO. Clear RX interrupt to
                * not involve interrupt handler when RX FIFO is empty.
                */
                SPIM_ClearRxInterruptSource(SPIM_INTR_RX_NOT_EMPTY);

                SPIM_EnableInt();
            }
            
        }

        /* Get and clear RX error mask */
        tmpStatus = (SPIM_GetRxInterruptSource() & SPIM_INTR_RX_ERR);
        SPIM_ClearRxInterruptSource(SPIM_INTR_RX_ERR);

        /* Puts together data and error status:
        * MP mode and accept address: 9th bit is set to notify mark.
        */
        rxData |= ((uint32) (tmpStatus << 8u));

        return (rxData);
    }


    #if !(SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1)
        /*******************************************************************************
        * Function Name: SPIM_UartSetRtsPolarity
        ****************************************************************************//**
        *
        *  Sets active polarity of RTS output signal.
        *  Only available for PSoC 4100 BLE / PSoC 4200 BLE / PSoC 4100M / PSoC 4200M /
        *  PSoC 4200L / PSoC 4000S / PSoC 4100S / PSoC Analog Coprocessor devices.
        *
        *  \param polarity: Active polarity of RTS output signal.
        *   - SPIM_UART_RTS_ACTIVE_LOW  - RTS signal is active low.
        *   - SPIM_UART_RTS_ACTIVE_HIGH - RTS signal is active high.
        *
        *******************************************************************************/
        void SPIM_UartSetRtsPolarity(uint32 polarity)
        {
            if(0u != polarity)
            {
                SPIM_UART_FLOW_CTRL_REG |= (uint32)  SPIM_UART_FLOW_CTRL_RTS_POLARITY;
            }
            else
            {
                SPIM_UART_FLOW_CTRL_REG &= (uint32) ~SPIM_UART_FLOW_CTRL_RTS_POLARITY;
            }
        }


        /*******************************************************************************
        * Function Name: SPIM_UartSetRtsFifoLevel
        ****************************************************************************//**
        *
        *  Sets level in the RX FIFO for RTS signal activation.
        *  While the RX FIFO has fewer entries than the RX FIFO level the RTS signal
        *  remains active, otherwise the RTS signal becomes inactive.
        *  Only available for PSoC 4100 BLE / PSoC 4200 BLE / PSoC 4100M / PSoC 4200M /
        *  PSoC 4200L / PSoC 4000S / PSoC 4100S / PSoC Analog Coprocessor devices.
        *
        *  \param level: Level in the RX FIFO for RTS signal activation.
        *   The range of valid level values is between 0 and RX FIFO depth - 1.
        *   Setting level value to 0 disables RTS signal activation.
        *
        *******************************************************************************/
        void SPIM_UartSetRtsFifoLevel(uint32 level)
        {
            uint32 uartFlowCtrl;

            uartFlowCtrl = SPIM_UART_FLOW_CTRL_REG;

            uartFlowCtrl &= ((uint32) ~SPIM_UART_FLOW_CTRL_TRIGGER_LEVEL_MASK); /* Clear level mask bits */
            uartFlowCtrl |= ((uint32) (SPIM_UART_FLOW_CTRL_TRIGGER_LEVEL_MASK & level));

            SPIM_UART_FLOW_CTRL_REG = uartFlowCtrl;
        }
    #endif /* !(SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1) */

#endif /* (SPIM_UART_RX_DIRECTION) */


#if(SPIM_UART_TX_DIRECTION)
    /*******************************************************************************
    * Function Name: SPIM_UartPutString
    ****************************************************************************//**
    *
    *  Places a NULL terminated string in the transmit buffer to be sent at the
    *  next available bus time.
    *  This function is blocking and waits until there is a space available to put
    *  requested data in transmit buffer.
    *
    *  \param string: pointer to the null terminated string array to be placed in the
    *   transmit buffer.
    *
    *******************************************************************************/
    void SPIM_UartPutString(const char8 string[])
    {
        uint32 bufIndex;

        bufIndex = 0u;

        /* Blocks the control flow until all data has been sent */
        while(string[bufIndex] != ((char8) 0))
        {
            SPIM_UartPutChar((uint32) string[bufIndex]);
            bufIndex++;
        }
    }


    /*******************************************************************************
    * Function Name: SPIM_UartPutCRLF
    ****************************************************************************//**
    *
    *  Places byte of data followed by a carriage return (0x0D) and line feed
    *  (0x0A) in the transmit buffer.
    *  This function is blocking and waits until there is a space available to put
    *  all requested data in transmit buffer.
    *
    *  \param txDataByte: the data to be transmitted.
    *
    *******************************************************************************/
    void SPIM_UartPutCRLF(uint32 txDataByte)
    {
        SPIM_UartPutChar(txDataByte);  /* Blocks control flow until all data has been sent */
        SPIM_UartPutChar(0x0Du);       /* Blocks control flow until all data has been sent */
        SPIM_UartPutChar(0x0Au);       /* Blocks control flow until all data has been sent */
    }


    #if !(SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1)
        /*******************************************************************************
        * Function Name: SPIMSCB_UartEnableCts
        ****************************************************************************//**
        *
        *  Enables usage of CTS input signal by the UART transmitter.
        *  Only available for PSoC 4100 BLE / PSoC 4200 BLE / PSoC 4100M / PSoC 4200M /
        *  PSoC 4200L / PSoC 4000S / PSoC 4100S / PSoC Analog Coprocessor devices.
        *
        *******************************************************************************/
        void SPIM_UartEnableCts(void)
        {
            SPIM_UART_FLOW_CTRL_REG |= (uint32)  SPIM_UART_FLOW_CTRL_CTS_ENABLE;
        }


        /*******************************************************************************
        * Function Name: SPIM_UartDisableCts
        ****************************************************************************//**
        *
        *  Disables usage of CTS input signal by the UART transmitter.
        *  Only available for PSoC 4100 BLE / PSoC 4200 BLE / PSoC 4100M / PSoC 4200M /
        *  PSoC 4200L / PSoC 4000S / PSoC 4100S / PSoC Analog Coprocessor devices.
        *
        *******************************************************************************/
        void SPIM_UartDisableCts(void)
        {
            SPIM_UART_FLOW_CTRL_REG &= (uint32) ~SPIM_UART_FLOW_CTRL_CTS_ENABLE;
        }


        /*******************************************************************************
        * Function Name: SPIM_UartSetCtsPolarity
        ****************************************************************************//**
        *
        *  Sets active polarity of CTS input signal.
        *  Only available for PSoC 4100 BLE / PSoC 4200 BLE / PSoC 4100M / PSoC 4200M /
        *  PSoC 4200L / PSoC 4000S / PSoC 4100S / PSoC Analog Coprocessor devices.
        *
        * \param
        * polarity: Active polarity of CTS output signal.
        *   - SPIM_UART_CTS_ACTIVE_LOW  - CTS signal is active low.
        *   - SPIM_UART_CTS_ACTIVE_HIGH - CTS signal is active high.
        *
        *******************************************************************************/
        void SPIM_UartSetCtsPolarity(uint32 polarity)
        {
            if (0u != polarity)
            {
                SPIM_UART_FLOW_CTRL_REG |= (uint32)  SPIM_UART_FLOW_CTRL_CTS_POLARITY;
            }
            else
            {
                SPIM_UART_FLOW_CTRL_REG &= (uint32) ~SPIM_UART_FLOW_CTRL_CTS_POLARITY;
            }
        }
    #endif /* !(SPIM_CY_SCBIP_V0 || SPIM_CY_SCBIP_V1) */


    /*******************************************************************************
    * Function Name: SPIM_UartSendBreakBlocking
    ****************************************************************************//**
    *
    * Sends a break condition (logic low) of specified width on UART TX line.
    * Blocks until break is completed. Only call this function when UART TX FIFO
    * and shifter are empty.
    *
    * \param breakWidth
    * Width of break condition. Valid range is 4 to 16 bits.
    *
    * \note
    * Before sending break all UART TX interrupt sources are disabled. The state
    * of UART TX interrupt sources is restored before function returns.
    *
    * \sideeffect
    * If this function is called while there is data in the TX FIFO or shifter that
    * data will be shifted out in packets the size of breakWidth.
    *
    *******************************************************************************/
    void SPIM_UartSendBreakBlocking(uint32 breakWidth)
    {
        uint32 txCtrlReg;
        uint32 txIntrReg;

        /* Disable all UART TX interrupt source and clear UART TX Done history */
        txIntrReg = SPIM_GetTxInterruptMode();
        SPIM_SetTxInterruptMode(0u);
        SPIM_ClearTxInterruptSource(SPIM_INTR_TX_UART_DONE);

        /* Store TX CTRL configuration */
        txCtrlReg = SPIM_TX_CTRL_REG;

        /* Set break width */
        SPIM_TX_CTRL_REG = (SPIM_TX_CTRL_REG & (uint32) ~SPIM_TX_CTRL_DATA_WIDTH_MASK) |
                                        SPIM_GET_TX_CTRL_DATA_WIDTH(breakWidth);

        /* Generate break */
        SPIM_TX_FIFO_WR_REG = 0u;

        /* Wait for break completion */
        while (0u == (SPIM_GetTxInterruptSource() & SPIM_INTR_TX_UART_DONE))
        {
        }

        /* Clear all UART TX interrupt sources to  */
        SPIM_ClearTxInterruptSource(SPIM_INTR_TX_ALL);

        /* Restore TX interrupt sources and data width */
        SPIM_TX_CTRL_REG = txCtrlReg;
        SPIM_SetTxInterruptMode(txIntrReg);
    }
#endif /* (SPIM_UART_TX_DIRECTION) */


#if (SPIM_UART_WAKE_ENABLE_CONST)
    /*******************************************************************************
    * Function Name: SPIM_UartSaveConfig
    ****************************************************************************//**
    *
    *  Clears and enables an interrupt on a falling edge of the Rx input. The GPIO
    *  interrupt does not track in the active mode, therefore requires to be
    *  cleared by this API.
    *
    *******************************************************************************/
    void SPIM_UartSaveConfig(void)
    {
    #if (SPIM_UART_RX_WAKEUP_IRQ)
        /* Set SKIP_START if requested (set by default). */
        if (0u != SPIM_skipStart)
        {
            SPIM_UART_RX_CTRL_REG |= (uint32)  SPIM_UART_RX_CTRL_SKIP_START;
        }
        else
        {
            SPIM_UART_RX_CTRL_REG &= (uint32) ~SPIM_UART_RX_CTRL_SKIP_START;
        }

        /* Clear RX GPIO interrupt status and pending interrupt in NVIC because
        * falling edge on RX line occurs while UART communication in active mode.
        * Enable interrupt: next interrupt trigger should wakeup device.
        */
        SPIM_CLEAR_UART_RX_WAKE_INTR;
        SPIM_RxWakeClearPendingInt();
        SPIM_RxWakeEnableInt();
    #endif /* (SPIM_UART_RX_WAKEUP_IRQ) */
    }


    /*******************************************************************************
    * Function Name: SPIM_UartRestoreConfig
    ****************************************************************************//**
    *
    *  Disables the RX GPIO interrupt. Until this function is called the interrupt
    *  remains active and triggers on every falling edge of the UART RX line.
    *
    *******************************************************************************/
    void SPIM_UartRestoreConfig(void)
    {
    #if (SPIM_UART_RX_WAKEUP_IRQ)
        /* Disable interrupt: no more triggers in active mode */
        SPIM_RxWakeDisableInt();
    #endif /* (SPIM_UART_RX_WAKEUP_IRQ) */
    }


    #if (SPIM_UART_RX_WAKEUP_IRQ)
        /*******************************************************************************
        * Function Name: SPIM_UART_WAKEUP_ISR
        ****************************************************************************//**
        *
        *  Handles the Interrupt Service Routine for the SCB UART mode GPIO wakeup
        *  event. This event is configured to trigger on a falling edge of the RX line.
        *
        *******************************************************************************/
        CY_ISR(SPIM_UART_WAKEUP_ISR)
        {
        #ifdef SPIM_UART_WAKEUP_ISR_ENTRY_CALLBACK
            SPIM_UART_WAKEUP_ISR_EntryCallback();
        #endif /* SPIM_UART_WAKEUP_ISR_ENTRY_CALLBACK */

            SPIM_CLEAR_UART_RX_WAKE_INTR;

        #ifdef SPIM_UART_WAKEUP_ISR_EXIT_CALLBACK
            SPIM_UART_WAKEUP_ISR_ExitCallback();
        #endif /* SPIM_UART_WAKEUP_ISR_EXIT_CALLBACK */
        }
    #endif /* (SPIM_UART_RX_WAKEUP_IRQ) */
#endif /* (SPIM_UART_RX_WAKEUP_IRQ) */


/* [] END OF FILE */
